"use strict";
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var __metadata = (this && this.__metadata) || function (k, v) {
    if (typeof Reflect === "object" && typeof Reflect.metadata === "function") return Reflect.metadata(k, v);
};
var __param = (this && this.__param) || function (paramIndex, decorator) {
    return function (target, key) { decorator(target, key, paramIndex); }
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.VendorsService = void 0;
const common_1 = require("@nestjs/common");
const typeorm_1 = require("@nestjs/typeorm");
const typeorm_2 = require("typeorm");
const vendor_entity_1 = require("../../entities/vendor.entity");
const product_entity_1 = require("../../entities/product.entity");
const category_entity_1 = require("../../entities/category.entity");
let VendorsService = class VendorsService {
    vendorRepo;
    productRepo;
    categoryRepo;
    constructor(vendorRepo, productRepo, categoryRepo) {
        this.vendorRepo = vendorRepo;
        this.productRepo = productRepo;
        this.categoryRepo = categoryRepo;
    }
    toVendorResponseDto(vendor) {
        if (!vendor) {
            throw new common_1.BadRequestException('Vendor cannot be null');
        }
        return {
            id: vendor.id,
            name: vendor.name,
            email: vendor.email,
            phone: vendor.phone,
            address: vendor.address,
            city: vendor.city,
            state: vendor.state,
            country: vendor.country,
            postalCode: vendor.postalCode,
            description: vendor.description,
            logo: vendor.logo,
            website: vendor.website,
            isApproved: vendor.isApproved,
            isActive: vendor.isActive,
            rating: vendor.rating,
            totalProducts: vendor.totalProducts,
            createdAt: vendor.createdAt,
            updatedAt: vendor.updatedAt,
            user: vendor.user ? {
                id: vendor.user.id,
                email: vendor.user.email,
                role: vendor.user.role,
            } : undefined,
            products: vendor.products?.map(product => ({
                id: product.id,
                name: product.name,
                price: parseFloat(product.price?.toString() || '0'),
                stock: product.stock,
                isFeatured: product.isFeatured,
                image: product.image,
            })) || [],
        };
    }
    async create(createVendorDto) {
        const existingVendor = await this.vendorRepo.findOne({
            where: { email: createVendorDto.email },
        });
        if (existingVendor) {
            throw new common_1.ConflictException('Vendor with this email already exists');
        }
        const existingVendorByName = await this.vendorRepo.findOne({
            where: { name: createVendorDto.name },
        });
        if (existingVendorByName) {
            throw new common_1.ConflictException('Vendor with this name already exists');
        }
        const vendor = this.vendorRepo.create({
            ...createVendorDto,
            isApproved: false,
            isActive: true,
        });
        const savedVendor = await this.vendorRepo.save(vendor);
        const completeVendor = await this.vendorRepo.findOne({
            where: { id: savedVendor.id },
            relations: ['user', 'products'],
        });
        if (!completeVendor) {
            throw new common_1.NotFoundException('Vendor not found after creation');
        }
        return this.toVendorResponseDto(completeVendor);
    }
    async findAll(query) {
        const { page = 1, limit = 10, search, isApproved, isActive, minRating, city, state, country, sortBy = 'createdAt', sortOrder = 'DESC', } = query;
        const skip = (page - 1) * limit;
        const where = {};
        if (isApproved !== undefined) {
            where.isApproved = isApproved;
        }
        if (isActive !== undefined) {
            where.isActive = isActive;
        }
        if (minRating !== undefined) {
            where.rating = (0, typeorm_2.MoreThan)(minRating);
        }
        if (city) {
            where.city = (0, typeorm_2.Like)(`%${city}%`);
        }
        if (state) {
            where.state = (0, typeorm_2.Like)(`%${state}%`);
        }
        if (country) {
            where.country = (0, typeorm_2.Like)(`%${country}%`);
        }
        if (search) {
            where.name = (0, typeorm_2.Like)(`%${search}%`);
        }
        const order = {};
        order[sortBy] = sortOrder.toUpperCase();
        const [vendors, total] = await this.vendorRepo.findAndCount({
            where,
            relations: ['user', 'products'],
            order,
            skip,
            take: limit,
        });
        const totalPages = Math.ceil(total / limit);
        return {
            vendors: vendors.map(vendor => this.toVendorResponseDto(vendor)),
            total,
            page,
            limit,
            totalPages,
        };
    }
    async findOne(id) {
        const vendor = await this.vendorRepo.findOne({
            where: { id },
            relations: ['user', 'products'],
        });
        if (!vendor) {
            throw new common_1.NotFoundException(`Vendor with ID ${id} not found`);
        }
        return this.toVendorResponseDto(vendor);
    }
    async findByEmail(email) {
        const vendor = await this.vendorRepo.findOne({
            where: { email },
            relations: ['user', 'products'],
        });
        if (!vendor) {
            throw new common_1.NotFoundException(`Vendor with email ${email} not found`);
        }
        return this.toVendorResponseDto(vendor);
    }
    async findByIds(ids) {
        if (!ids || ids.length === 0) {
            return [];
        }
        const vendors = await this.vendorRepo.find({
            where: { id: (0, typeorm_2.In)(ids) },
            relations: ['user', 'products'],
        });
        return vendors.map(vendor => this.toVendorResponseDto(vendor));
    }
    async update(id, updateVendorDto) {
        const vendor = await this.vendorRepo.findOne({
            where: { id },
            relations: ['user', 'products'],
        });
        if (!vendor) {
            throw new common_1.NotFoundException(`Vendor with ID ${id} not found`);
        }
        if (updateVendorDto.email && updateVendorDto.email !== vendor.email) {
            const existingVendor = await this.vendorRepo.findOne({
                where: { email: updateVendorDto.email },
            });
            if (existingVendor && existingVendor.id !== id) {
                throw new common_1.ConflictException('Vendor with this email already exists');
            }
        }
        if (updateVendorDto.name && updateVendorDto.name !== vendor.name) {
            const existingVendor = await this.vendorRepo.findOne({
                where: { name: updateVendorDto.name },
            });
            if (existingVendor && existingVendor.id !== id) {
                throw new common_1.ConflictException('Vendor with this name already exists');
            }
        }
        Object.assign(vendor, updateVendorDto);
        const updatedVendor = await this.vendorRepo.save(vendor);
        const completeVendor = await this.vendorRepo.findOne({
            where: { id: updatedVendor.id },
            relations: ['user', 'products'],
        });
        if (!completeVendor) {
            throw new common_1.NotFoundException(`Vendor with ID ${updatedVendor.id} not found after update`);
        }
        return this.toVendorResponseDto(completeVendor);
    }
    async remove(id) {
        const vendor = await this.vendorRepo.findOne({
            where: { id },
            relations: ['products'],
        });
        if (!vendor) {
            throw new common_1.NotFoundException(`Vendor with ID ${id} not found`);
        }
        if (vendor.products && vendor.products.length > 0) {
            throw new common_1.ConflictException('Cannot delete vendor with existing products. Deactivate instead.');
        }
        await this.vendorRepo.remove(vendor);
    }
    async approveVendor(id) {
        const vendor = await this.vendorRepo.findOne({ where: { id } });
        if (!vendor) {
            throw new common_1.NotFoundException(`Vendor with ID ${id} not found`);
        }
        if (vendor.isApproved) {
            throw new common_1.BadRequestException('Vendor is already approved');
        }
        vendor.isApproved = true;
        vendor.isActive = true;
        const updatedVendor = await this.vendorRepo.save(vendor);
        const completeVendor = await this.vendorRepo.findOne({
            where: { id: updatedVendor.id },
            relations: ['user', 'products'],
        });
        if (!completeVendor) {
            throw new common_1.NotFoundException(`Vendor with ID ${updatedVendor.id} not found after approval`);
        }
        return this.toVendorResponseDto(completeVendor);
    }
    async rejectVendor(id) {
        const vendor = await this.vendorRepo.findOne({ where: { id } });
        if (!vendor) {
            throw new common_1.NotFoundException(`Vendor with ID ${id} not found`);
        }
        vendor.isApproved = false;
        const updatedVendor = await this.vendorRepo.save(vendor);
        const completeVendor = await this.vendorRepo.findOne({
            where: { id: updatedVendor.id },
            relations: ['user', 'products'],
        });
        if (!completeVendor) {
            throw new common_1.NotFoundException(`Vendor with ID ${updatedVendor.id} not found after rejection`);
        }
        return this.toVendorResponseDto(completeVendor);
    }
    async toggleActive(id) {
        const vendor = await this.vendorRepo.findOne({ where: { id } });
        if (!vendor) {
            throw new common_1.NotFoundException(`Vendor with ID ${id} not found`);
        }
        vendor.isActive = !vendor.isActive;
        const updatedVendor = await this.vendorRepo.save(vendor);
        const completeVendor = await this.vendorRepo.findOne({
            where: { id: updatedVendor.id },
            relations: ['user', 'products'],
        });
        if (!completeVendor) {
            throw new common_1.NotFoundException(`Vendor with ID ${updatedVendor.id} not found after toggle`);
        }
        return this.toVendorResponseDto(completeVendor);
    }
    async updateRating(id, newRating) {
        if (newRating < 0 || newRating > 5) {
            throw new common_1.BadRequestException('Rating must be between 0 and 5');
        }
        const vendor = await this.vendorRepo.findOne({ where: { id } });
        if (!vendor) {
            throw new common_1.NotFoundException(`Vendor with ID ${id} not found`);
        }
        vendor.rating = newRating;
        const updatedVendor = await this.vendorRepo.save(vendor);
        const completeVendor = await this.vendorRepo.findOne({
            where: { id: updatedVendor.id },
            relations: ['user', 'products'],
        });
        if (!completeVendor) {
            throw new common_1.NotFoundException(`Vendor with ID ${updatedVendor.id} not found after rating update`);
        }
        return this.toVendorResponseDto(completeVendor);
    }
    async getVendorProducts(id) {
        const vendor = await this.vendorRepo.findOne({ where: { id } });
        if (!vendor) {
            throw new common_1.NotFoundException(`Vendor with ID ${id} not found`);
        }
        const products = await this.productRepo.find({
            where: { vendorId: id },
            relations: ['category', 'variations'],
            order: { createdAt: 'DESC' },
        });
        return products.map(product => ({
            id: product.id,
            name: product.name,
            price: parseFloat(product.price?.toString() || '0'),
            stock: product.stock,
            isFeatured: product.isFeatured,
            image: product.image,
            category: product.category ? {
                id: product.category.id,
                name: product.category.name,
            } : null,
            variations: product.variations?.map(variation => ({
                id: variation.id,
                name: variation.name,
                price: parseFloat(variation.price?.toString() || '0'),
                stock: variation.stock,
            })) || [],
        }));
    }
    async createVendorProduct(vendorId, createProductDto) {
        const vendor = await this.vendorRepo.findOne({ where: { id: vendorId } });
        if (!vendor) {
            throw new common_1.NotFoundException(`Vendor with ID ${vendorId} not found`);
        }
        if (!vendor.isApproved) {
            throw new common_1.BadRequestException('Cannot create product for unapproved vendor');
        }
        const existingProduct = await this.productRepo.findOne({
            where: {
                name: createProductDto.name,
                vendorId: vendorId
            },
        });
        if (existingProduct) {
            throw new common_1.ConflictException('Product with this name already exists for your vendor account');
        }
        if (createProductDto.categoryId) {
            const category = await this.categoryRepo.findOne({
                where: { id: createProductDto.categoryId },
            });
            if (!category) {
                throw new common_1.NotFoundException(`Category with ID ${createProductDto.categoryId} not found`);
            }
        }
        const product = this.productRepo.create({
            ...createProductDto,
            vendorId: vendorId,
        });
        const savedProduct = await this.productRepo.save(product);
        const completeProduct = await this.productRepo.findOne({
            where: { id: savedProduct.id },
            relations: ['category', 'vendor', 'variations'],
        });
        if (!completeProduct) {
            throw new common_1.NotFoundException(`Product with ID ${savedProduct.id} not found after creation`);
        }
        return {
            id: completeProduct.id,
            name: completeProduct.name,
            description: completeProduct.description,
            price: parseFloat(completeProduct.price?.toString() || '0'),
            image: completeProduct.image,
            stock: completeProduct.stock,
            isFeatured: completeProduct.isFeatured,
            category: completeProduct.category ? {
                id: completeProduct.category.id,
                name: completeProduct.category.name,
            } : null,
            variations: completeProduct.variations?.map(variation => ({
                id: variation.id,
                name: variation.name,
                price: parseFloat(variation.price?.toString() || '0'),
                stock: variation.stock,
            })) || [],
        };
    }
    async getVendorProductById(vendorId, productId) {
        const product = await this.productRepo.findOne({
            where: {
                id: productId,
                vendorId: vendorId
            },
            relations: ['category', 'vendor', 'variations'],
        });
        if (!product) {
            throw new common_1.NotFoundException('Product not found or you do not have permission to access it');
        }
        return {
            id: product.id,
            name: product.name,
            description: product.description,
            price: parseFloat(product.price?.toString() || '0'),
            image: product.image,
            stock: product.stock,
            isFeatured: product.isFeatured,
            category: product.category ? {
                id: product.category.id,
                name: product.category.name,
            } : null,
            variations: product.variations?.map(variation => ({
                id: variation.id,
                name: variation.name,
                price: parseFloat(variation.price?.toString() || '0'),
                stock: variation.stock,
            })) || [],
        };
    }
    async updateVendorProduct(vendorId, productId, updateProductDto) {
        const product = await this.productRepo.findOne({
            where: {
                id: productId,
                vendorId: vendorId
            },
        });
        if (!product) {
            throw new common_1.NotFoundException('Product not found or you do not have permission to update it');
        }
        if (updateProductDto.name && updateProductDto.name !== product.name) {
            const existingProduct = await this.productRepo.findOne({
                where: {
                    name: updateProductDto.name,
                    vendorId: vendorId
                },
            });
            if (existingProduct && existingProduct.id !== productId) {
                throw new common_1.ConflictException('Product with this name already exists for your vendor account');
            }
        }
        if (updateProductDto.categoryId) {
            const category = await this.categoryRepo.findOne({
                where: { id: updateProductDto.categoryId },
            });
            if (!category) {
                throw new common_1.NotFoundException(`Category with ID ${updateProductDto.categoryId} not found`);
            }
        }
        Object.assign(product, updateProductDto);
        const updatedProduct = await this.productRepo.save(product);
        const completeProduct = await this.productRepo.findOne({
            where: { id: updatedProduct.id },
            relations: ['category', 'vendor', 'variations'],
        });
        if (!completeProduct) {
            throw new common_1.NotFoundException(`Product with ID ${updatedProduct.id} not found after update`);
        }
        return {
            id: completeProduct.id,
            name: completeProduct.name,
            description: completeProduct.description,
            price: parseFloat(completeProduct.price?.toString() || '0'),
            image: completeProduct.image,
            stock: completeProduct.stock,
            isFeatured: completeProduct.isFeatured,
            category: completeProduct.category ? {
                id: completeProduct.category.id,
                name: completeProduct.category.name,
            } : null,
            variations: completeProduct.variations?.map(variation => ({
                id: variation.id,
                name: variation.name,
                price: parseFloat(variation.price?.toString() || '0'),
                stock: variation.stock,
            })) || [],
        };
    }
    async deleteVendorProduct(vendorId, productId) {
        const product = await this.productRepo.findOne({
            where: {
                id: productId,
                vendorId: vendorId
            },
            relations: ['orderDetails'],
        });
        if (!product) {
            throw new common_1.NotFoundException('Product not found or you do not have permission to delete it');
        }
        if (Array.isArray(product.orderDetails) && product.orderDetails.length > 0) {
            throw new common_1.ConflictException('Cannot delete product with order history. Deactivate it instead.');
        }
        await this.productRepo.remove(product);
    }
    async getVendorOrders(vendorId, query = {}) {
        const { page = 1, limit = 10, status } = query;
        const skip = (page - 1) * limit;
        const orderQuery = this.productRepo
            .createQueryBuilder('product')
            .innerJoinAndSelect('product.orderDetails', 'orderDetails')
            .innerJoinAndSelect('orderDetails.order', 'order')
            .where('product.vendorId = :vendorId', { vendorId });
        if (status) {
            orderQuery.andWhere('order.status = :status', { status });
        }
        const [orders, total] = await orderQuery
            .orderBy('order.createdAt', 'DESC')
            .skip(skip)
            .take(limit)
            .getManyAndCount();
        const totalPages = Math.ceil(total / limit);
        return {
            orders: orders.flatMap(product => Array.isArray(product.orderDetails)
                ? product.orderDetails.map(orderDetail => ({
                    id: orderDetail.order.id,
                    orderNumber: orderDetail.order.orderNumber,
                    status: orderDetail.order.status,
                    total: orderDetail.order.total,
                    createdAt: orderDetail.order.createdAt,
                    product: {
                        id: product.id,
                        name: product.name,
                        quantity: orderDetail.quantity,
                        price: orderDetail.price,
                    }
                }))
                : []),
            total,
            page,
            limit,
            totalPages,
        };
    }
    async getVendorDashboard(vendorId) {
        const stats = await this.getVendorStats(vendorId);
        const recentOrders = await this.getVendorOrders(vendorId, { limit: 5 });
        const lowStockProducts = await this.productRepo.find({
            where: {
                vendorId: vendorId,
                stock: (0, typeorm_2.LessThanOrEqual)(10)
            },
            take: 5,
            order: { stock: 'ASC' }
        });
        const recentProducts = await this.productRepo.find({
            where: { vendorId: vendorId },
            order: { createdAt: 'DESC' },
            take: 5,
        });
        return {
            stats,
            recentOrders: recentOrders.orders,
            lowStockProducts: lowStockProducts.map(product => ({
                id: product.id,
                name: product.name,
                stock: product.stock,
                image: product.image,
            })),
            recentProducts: recentProducts.map(product => ({
                id: product.id,
                name: product.name,
                price: parseFloat(product.price?.toString() || '0'),
                stock: product.stock,
                image: product.image,
            })),
        };
    }
    async getVendorStats(id) {
        const vendor = await this.vendorRepo.findOne({ where: { id } });
        if (!vendor) {
            throw new common_1.NotFoundException(`Vendor with ID ${id} not found`);
        }
        const totalProducts = await this.productRepo.count({ where: { vendorId: id } });
        const featuredProducts = await this.productRepo.count({
            where: { vendorId: id, isFeatured: true }
        });
        const outOfStockProducts = await this.productRepo.count({
            where: { vendorId: id, stock: 0 }
        });
        const lowStockProducts = await this.productRepo.count({
            where: { vendorId: id, stock: (0, typeorm_2.LessThanOrEqual)(10) }
        });
        const revenueResult = await this.productRepo
            .createQueryBuilder('product')
            .select('SUM(product.price)', 'totalRevenue')
            .where('product.vendorId = :vendorId', { vendorId: id })
            .getRawOne();
        return {
            totalProducts,
            featuredProducts,
            outOfStockProducts,
            lowStockProducts,
            totalRevenue: parseFloat(revenueResult?.totalRevenue || '0'),
            rating: vendor.rating,
            isApproved: vendor.isApproved,
            isActive: vendor.isActive,
        };
    }
    async getPendingVendors() {
        const vendors = await this.vendorRepo.find({
            where: { isApproved: false },
            relations: ['user', 'products'],
            order: { createdAt: 'ASC' },
        });
        return vendors.map(vendor => this.toVendorResponseDto(vendor));
    }
    async getTopVendors(limit = 10) {
        const vendors = await this.vendorRepo.find({
            where: { isApproved: true, isActive: true, rating: (0, typeorm_2.MoreThan)(4) },
            relations: ['user', 'products'],
            order: { rating: 'DESC', totalProducts: 'DESC' },
            take: limit,
        });
        return vendors.map(vendor => this.toVendorResponseDto(vendor));
    }
    async searchVendors(searchTerm, limit = 20) {
        if (!searchTerm || searchTerm.trim().length === 0) {
            throw new common_1.BadRequestException('Search term is required');
        }
        const vendors = await this.vendorRepo.find({
            where: [
                { name: (0, typeorm_2.Like)(`%${searchTerm}%`) },
                { email: (0, typeorm_2.Like)(`%${searchTerm}%`) },
                { city: (0, typeorm_2.Like)(`%${searchTerm}%`) },
                { state: (0, typeorm_2.Like)(`%${searchTerm}%`) },
            ],
            relations: ['user', 'products'],
            order: { name: 'ASC' },
            take: limit,
        });
        return vendors.map(vendor => this.toVendorResponseDto(vendor));
    }
    async getVendorsByLocation(city, state, country) {
        const where = { isApproved: true, isActive: true };
        if (city) {
            where.city = (0, typeorm_2.Like)(`%${city}%`);
        }
        if (state) {
            where.state = (0, typeorm_2.Like)(`%${state}%`);
        }
        if (country) {
            where.country = (0, typeorm_2.Like)(`%${country}%`);
        }
        const vendors = await this.vendorRepo.find({
            where,
            relations: ['user', 'products'],
            order: { name: 'ASC' },
        });
        return vendors.map(vendor => this.toVendorResponseDto(vendor));
    }
};
exports.VendorsService = VendorsService;
exports.VendorsService = VendorsService = __decorate([
    (0, common_1.Injectable)(),
    __param(0, (0, typeorm_1.InjectRepository)(vendor_entity_1.Vendor)),
    __param(1, (0, typeorm_1.InjectRepository)(product_entity_1.Product)),
    __param(2, (0, typeorm_1.InjectRepository)(category_entity_1.Category)),
    __metadata("design:paramtypes", [typeorm_2.Repository,
        typeorm_2.Repository,
        typeorm_2.Repository])
], VendorsService);
//# sourceMappingURL=vendor.service.js.map